'use strict';

/* --------------------------------------------------------------
 emails_paginator.js 2018-06-07
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Emails Paginator Controller
 *
 * This controller will handle the main table paginator operations of the admin/emails page.
 *
 * @module Controllers/emails_paginator
 */
gx.controllers.module('emails_paginator', [gx.source + '/libs/emails', gx.source + '/libs/button_dropdown', 'loading_spinner', 'modal'],

/** @lends module:Controllers/emails_paginator */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Reference
     *
     * @type {object}
     */
    $this = $(this),


    /**
     * Table Selector
     *
     * @type {object}
     */
    $table = $('#emails-table'),


    /**
     * Attachments Size Selector
     *
     * @type {object}
     */
    $attachmentsSize = $('#attachments-size'),


    /**
     * Default Module Options
     *
     * @type {object}
     */
    defaults = {},


    /**
     * Final Module Options
     *
     * @type {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // EVENT HANDLERS
    // ------------------------------------------------------------------------

    /**
     * Refresh page data.
     *
     * @param {object} event Contains event information.
     */
    var _onRefreshData = function _onRefreshData(event) {
        $table.DataTable().ajax.reload();
        jse.libs.emails.getAttachmentsSize($attachmentsSize);
    };

    /**
     * Change current page length.
     *
     * @param {object} event Contains the event data.
     */
    var _onTableLengthChange = function _onTableLengthChange(event) {
        var length = $this.find('#display-records').val();
        $table.DataTable().page.len(length).draw();
    };

    /**
     * Open handle attachments modal window.
     *
     * @param {object} event Contains event information.
     */
    var _onHandleAttachments = function _onHandleAttachments(event) {
        var $attachmentsModal = $('#attachments-modal');

        // Reset modal state.
        $attachmentsModal.find('#removal-date').val('').datepicker({
            maxDate: new Date()
        });
        $(document).find('.ui-datepicker').not('.gx-container').addClass('gx-container');

        // Display modal to the user.
        $attachmentsModal.dialog({
            title: jse.core.lang.translate('handle_attachments', 'emails'),
            width: 400,
            modal: true,
            dialogClass: 'gx-container',
            closeOnEscape: true
        });

        $('#close-modal').on('click', function () {
            $($attachmentsModal).dialog('close');
        });
    };

    /**
     * Open handle attachments modal window.
     *
     * @param {object} event Contains event information.
     */
    var _onHandleOldEmails = function _onHandleOldEmails(event) {
        var $deleteOldEmailsModal = $('#delete-old-emails-modal');

        // Reset modal state.
        $deleteOldEmailsModal.find('#email-removal-date').val('').datepicker({
            maxDate: new Date()
        });
        $(document).find('.ui-datepicker').not('.gx-container').addClass('gx-container');

        // Display modal to the user.
        $deleteOldEmailsModal.dialog({
            title: jse.core.lang.translate('handle_old_emails', 'emails'),
            width: 400,
            modal: true,
            dialogClass: 'gx-container',
            closeOnEscape: true
        });

        $('#close-old-email-modal').on('click', function () {
            $($deleteOldEmailsModal).dialog('close');
        });
    };

    /**
     * Execute the delete operation for the selected email records.
     *
     * @param {object} event Contains the event information.
     */
    var _onBulkDelete = function _onBulkDelete(event) {
        // Check if there are table rows selected.
        if ($table.find('tr td input:checked').length === 0 || $('#bulk-action').val() === '') {
            return; // No selected records, exit method.
        }

        // Get selected rows data - create a new email collection.
        var collection = jse.libs.emails.getSelectedEmails($table);

        // Display confirmation modal to the user.
        jse.libs.modal.message({
            title: jse.core.lang.translate('bulk_action', 'admin_labels'),
            content: jse.core.lang.translate('prompt_delete_collection', 'emails'),
            buttons: [{
                text: jse.core.lang.translate('no', 'lightbox_buttons'),
                click: function click() {
                    $(this).dialog('close');
                }
            }, {
                text: jse.core.lang.translate('yes', 'lightbox_buttons'),
                click: function click() {
                    jse.libs.emails.deleteCollection(collection).done(function (response) {
                        $table.DataTable().ajax.reload();
                        jse.libs.emails.getAttachmentsSize($attachmentsSize);
                    }).fail(function (response) {
                        var title = jse.core.lang.translate('error', 'messages');

                        jse.libs.modal.message({
                            title: title,
                            content: response.message
                        });
                    });

                    $(this).dialog('close');
                    $table.find('input[type=checkbox]').prop('checked', false);
                }
            }]
        });
    };

    /**
     * Execute the send operation for the selected email records.
     *
     * @param {object} event Contains the event information.
     */
    var _onBulkSend = function _onBulkSend(event) {
        // Check if there are table rows selected.
        if ($table.find('tr td input:checked').length === 0 || $('#bulk-action').val() === '') {
            return; // No selected records, exit method.
        }

        // Get selected rows data - create a new email collection.
        var collection = jse.libs.emails.getSelectedEmails($table);

        // Display confirmation modal to the user.
        jse.libs.modal.message({
            title: jse.core.lang.translate('bulk_action', 'admin_labels'),
            content: jse.core.lang.translate('prompt_send_collection', 'emails'),
            buttons: [{
                text: jse.core.lang.translate('no', 'lightbox_buttons'),
                click: function click() {
                    $(this).dialog('close');
                }
            }, {
                text: jse.core.lang.translate('yes', 'lightbox_buttons'),
                click: function click() {
                    jse.libs.emails.sendCollection(collection).done(function (response) {
                        $table.DataTable().ajax.reload();
                        jse.libs.emails.getAttachmentsSize($attachmentsSize);
                    }).fail(function (response) {
                        var title = jse.core.lang.translate('error', 'messages');

                        jse.libs.modal.message({
                            title: title,
                            content: response.message
                        });
                    });

                    $(this).dialog('close');
                    $table.find('input[type=checkbox]').prop('checked', false);
                }
            }]
        });
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    /**
     * Initialize method of the module, called by the engine.
     */
    module.init = function (done) {
        // Bind paginator event handlers.
        $this.on('click', '#refresh-table', _onRefreshData).on('change', '#display-records', _onTableLengthChange);
        $('body').on('click', '#handle-attachments', _onHandleAttachments);
        $('body').on('click', '#handle-old-emails', _onHandleOldEmails);

        var $dropdown = $this.find('.bulk-action');
        jse.libs.button_dropdown.mapAction($dropdown, 'bulk_send_selected', 'emails', _onBulkSend);
        jse.libs.button_dropdown.mapAction($dropdown, 'bulk_delete_selected', 'emails', _onBulkDelete);

        // Get current attachments size.
        jse.libs.emails.getAttachmentsSize($attachmentsSize);

        done();
    };

    // Return module object to module engine.
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
